<?php

namespace App\Http\Controllers;

use App\File;
use App\Task;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Storage;

class TasksController extends Controller
{
    /**
     * Tasks list
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function list(Request $request)
    {
        $query = $request->get('q');
        $doneHidden = Cookie::get('done_hidden');
        $tasks = Task::orderBy('done', 'ASC')->orderBy('done_date', 'DESC')->orderBy('created_at', 'DESC');

        if($doneHidden) {
            $tasks = $tasks->where('done', '=', 0);
        }

        if(!empty($query)){
            $tasks = $tasks->where('title', 'LIKE', '%'.$query.'%')->orWhere('description', 'LIKE', '%'.$query.'%');
        }

        $tasks = $tasks->get();

        return view('tasks.list')->with(compact('tasks'));
    }

    /**
     * Task view
     *
     * @param $id
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function view($id)
    {
        $task = Task::where('id', '=', $id)->with(['creator'])->first();

        return view('tasks.view')->with(compact('task'));
    }

    /**
     * Task create
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        return view('tasks.create');
    }

    /**
     * Task store
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'min:5|max:250|required',
            'description' => 'min:5|required',
            'files.*' => 'nullable|mimes:pdf,xml,jpg|max:5120',
        ]);

        $task = new Task;
        $task->creator_id = Auth::id();
        $task->title = $request->get('title');
        $task->description = $request->get('description');

        $task->save();

        if ($request->hasFile('files')) {
            $files = $request->file('files');
            foreach ($files as $file) {
                $uploadFile = $file->getClientOriginalName();
                $uploadFileName = pathinfo($uploadFile, PATHINFO_FILENAME);
                $uploadFileExt = $file->getClientOriginalExtension();
                $uploadFileToStore = $uploadFileName . '_' . mt_rand(1000, 9999) . time() . '.' . $uploadFileExt;

                $path = $file->storeAs('public/upload/' . $task->id, $uploadFileToStore);

                $uploadFile = new File();
                $uploadFile->task_id = $task->id;
                $uploadFile->name = $uploadFileToStore;
                $uploadFile->path = 'upload/' . $task->id;
                $uploadFile->data = json_encode($file->getFileInfo());
                $uploadFile->ext = $uploadFileExt;

                $errors = [];
                if (!$uploadFile->save()) {
                    $errors[] = $uploadFile->id;
                }
            }
        }

        return redirect()->route('taskView', $task->id)->with(['success' => 'Sikeres feladat felvétel']);
    }

    /**
     * Task edit
     *
     * @param $id
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($id)
    {
        $task = Task::where('id', '=', $id)->first();

        return view('tasks.edit')->with(compact('task'));
    }

    /**
     * Task update
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request)
    {
        $this->validate($request, [
            'title' => 'min:5|max:250|required',
            'description' => 'min:5|required',
            'files.*' => 'nullable|mimes:pdf,xml,jpg|max:5120',
        ]);

        $task = Task::where('id', '=', $request->get('task_id'))->first();
        $task = $this->_patchEntity($task, $request);

        $task->save();

        if ($request->hasFile('files')) {
            $files = $request->file('files');
            foreach ($files as $file) {
                $uploadFile = $file->getClientOriginalName();
                $uploadFileName = pathinfo($uploadFile, PATHINFO_FILENAME);
                $uploadFileExt = $file->getClientOriginalExtension();
                $uploadFileToStore = $uploadFileName . '_' . mt_rand(1000, 9999) . time() . '.' . $uploadFileExt;

                $path = $file->storeAs('public/upload/' . $task->id, $uploadFileToStore);

                $uploadFile = new File();
                $uploadFile->task_id = $task->id;
                $uploadFile->name = $uploadFileToStore;
                $uploadFile->path = 'upload/' . $task->id;
                $uploadFile->data = json_encode($file->getFileInfo());
                $uploadFile->ext = $uploadFileExt;

                $errors = [];
                if (!$uploadFile->save()) {
                    $errors[] = $uploadFile->id;
                }
            }
        }

        return redirect()->route('taskView', $task->id)->with(['success' => 'Sikeres feladat szerkesztés']);
    }

    /**
     * Task close
     *
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function close($id)
    {
        $task = Task::where('id', '=', $id)->first();

        $task->done = 1;
        $task->done_date = now();

        if ($task->save()) {
            return redirect()->back()->with(['success' => 'Sikeres feladat lezárás']);
        }
    }

    /**
     * Task delete
     *
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function delete($id)
    {
        $task = Task::where('id', '=', $id)->first();

        if (!empty($task->files)) {
            foreach ($task->files as $file) {
                $file->delete();
                Storage::delete('public/' . $file->path . '/' . $file->name);
            }
        }

        if ($task->delete()) {
            return redirect()->route('taskList')->with(['success' => 'Sikeres feladat törlés!']);
        }
    }

    /**
     * Task hide
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function hide(){
        $hidden = Cookie::get('done_hidden');

        if($hidden){
            Cookie::queue('done_hidden', false, 60*24*7);
        } else {
            Cookie::queue('done_hidden', true, 60*24*7);
        }

        return redirect()->back();
    }
}