<?php

namespace App\Http\Controllers;

use App\File;
use App\Task;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;

class TasksController extends Controller
{
    public function list(Request $request)
    {
        $query = $request->get('q');
        $doneHidden = Cookie::get('done_hidden');
        $tasks = Task::orderBy('done', 'ASC')->orderBy('done_date', 'DESC')->orderBy('created_at', 'DESC');

        if($doneHidden) {
            $tasks = $tasks->where('done', '=', 0);
        }

        if(!empty($query)){
            $tasks = $tasks->where('title', 'LIKE', '%'.$query.'%')->orWhere('description', 'LIKE', '%'.$query.'%');
        }

        $tasks = $tasks->get();

        return view('tasks.list')->with(compact('tasks'));
    }

    public function view($id)
    {
        $task = Task::where('id', '=', $id)->with(['creator'])->first();

        return view('tasks.view')->with(compact('task'));
    }

    public function create()
    {
        return view('tasks.create');
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'min:5|max:250|required',
            'description' => 'min:5|required',
            'files.*' => 'nullable|mimes:pdf,xml,jpg|max:5120',
        ]);

        $task = new Task;
        $task->creator_id = Auth::id();
        $task->title = $request->get('title');
        $task->description = $request->get('description');

        $task->save();

        if ($request->hasFile('files')) {
            $files = $request->file('files');
            foreach ($files as $file) {
                $uploadFile = $file->getClientOriginalName();
                $uploadFileName = pathinfo($uploadFile, PATHINFO_FILENAME);
                $uploadFileExt = $file->getClientOriginalExtension();
                $uploadFileToStore = $uploadFileName . '_' . mt_rand(1000, 9999) . time() . '.' . $uploadFileExt;

                $path = $file->storeAs('public/upload/' . $task->id, $uploadFileToStore);

                $uploadFile = new File();
                $uploadFile->task_id = $task->id;
                $uploadFile->name = $uploadFileToStore;
                $uploadFile->path = 'upload/' . $task->id;
                $uploadFile->data = json_encode($file->getFileInfo());
                $uploadFile->ext = $uploadFileExt;

                $errors = [];
                if (!$uploadFile->save()) {
                    $errors[] = $uploadFile->id;
                }
            }
        }

        return redirect()->route('taskView', $task->id)->with(['success' => 'Sikeres feladat felvétel']);
    }

    public function edit($id)
    {
        $task = Task::where('id', '=', $id)->first();

        return view('tasks.edit')->with(compact('task'));
    }

    public function update(Request $request)
    {
        $this->validate($request, [
            'title' => 'min:5|max:250|required',
            'description' => 'min:5|required',
            'files.*' => 'nullable|mimes:pdf,xml,jpg|max:5120',
        ]);

        $task = Task::where('id', '=', $request->get('task_id'))->first();
        $task = $this->_patchEntity($task, $request);

        $task->save();

        if ($request->hasFile('files')) {
            $files = $request->file('files');
            foreach ($files as $file) {
                $uploadFile = $file->getClientOriginalName();
                $uploadFileName = pathinfo($uploadFile, PATHINFO_FILENAME);
                $uploadFileExt = $file->getClientOriginalExtension();
                $uploadFileToStore = $uploadFileName . '_' . mt_rand(1000, 9999) . time() . '.' . $uploadFileExt;

                $path = $file->storeAs('public/upload/' . $task->id, $uploadFileToStore);

                $uploadFile = new File();
                $uploadFile->task_id = $task->id;
                $uploadFile->name = $uploadFileToStore;
                $uploadFile->path = 'upload/' . $task->id;
                $uploadFile->data = json_encode($file->getFileInfo());
                $uploadFile->ext = $uploadFileExt;

                $errors = [];
                if (!$uploadFile->save()) {
                    $errors[] = $uploadFile->id;
                }
            }
        }

        return redirect()->route('taskView', $task->id)->with(['success' => 'Sikeres feladat szerkesztés']);
    }

    public function close($id)
    {
        $task = Task::where('id', '=', $id)->first();

        $task->done = 1;
        $task->done_date = now();

        if ($task->save()) {
            return redirect()->back()->with(['success' => 'Sikeres feladat lezárás']);
        }
    }

    public function delete($id)
    {
        $task = Task::where('id', '=', $id)->first();

        if (!empty($task->files)) {
            foreach ($task->files as $file) {
                $file->delete();
            }
        }

        if ($task->delete()) {
            return redirect()->route('taskList')->with(['success' => 'Sikeres feladat törlés!']);
        }
    }

    public function hide(){
        $hidden = Cookie::get('done_hidden');

        if($hidden){
            Cookie::queue('done_hidden', false, 60*24*7);
        } else {
            Cookie::queue('done_hidden', true, 60*24*7);
        }

        return redirect()->back();
    }
}